
import junit.framework.TestCase;
import org.apache.ojb.broker.PBKey;
import org.apache.ojb.broker.PersistenceBroker;
import org.apache.ojb.broker.PersistenceBrokerFactory;
import org.apache.ojb.broker.Identity;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.Query;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.broker.query.ReportQueryByCriteria;
import org.apache.ojb.otm.OTMConnection;
import org.apache.ojb.otm.core.BaseConnection;
import org.apache.ojb.otm.core.Transaction;
import org.apache.ojb.otm.kit.SimpleKit;
import org.apache.ojb.otm.lock.LockingException;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;

public class TestDeliveries extends TestCase {
    private SimpleKit kit = SimpleKit.getInstance();
    private PBKey key = PersistenceBrokerFactory.getDefaultKey();

    public void setUp() {
        BaseConnection conn = (BaseConnection) kit.acquireConnection(key);
        PersistenceBroker pb = conn.getKernelBroker();
        pb.deleteByQuery(QueryFactory.newQuery(Beer.class, new Criteria()));
        pb.deleteByQuery(QueryFactory.newQuery(Truck.class, new Criteria()));
        pb.deleteByQuery(QueryFactory.newQuery(Pub.class, new Criteria()));
        conn.close();
    }


    public void testPersistTruck() throws LockingException {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        truck.load(new Beer[] { new DomesticBeer(), new DomesticBeer() });
        conn.makePersistent(truck);
        tx.commit();

        Identity id = conn.getIdentity(truck);
        ((BaseConnection)conn).getKernelBroker().clearCache();

        tx = kit.getTransaction(conn);
        tx.begin();
        truck = (Truck<Beer>) conn.getObjectByIdentity(id);
        System.err.println(truck.getId());
        tx.commit();

        for (Iterator<Beer> iterator = truck.unload().iterator(); iterator.hasNext();) {
            Beer beer = iterator.next();
            System.err.println(beer.getBrand());
        }

        conn.close();
    }

    public void testFillHerUp() throws Exception {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        Beer schlitz = new DomesticBeer();
        Beer more_schlitz = new DomesticBeer();
        Beer los_of_schlitz = new DomesticBeer();
        truck.load(new Beer[]{schlitz, more_schlitz, los_of_schlitz});
        conn.makePersistent(truck);
        tx.commit();
        conn.close();
    }

    public void testDumpInTheStreet() throws Exception {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);

        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        Beer schlitz = new DomesticBeer();
        Beer more_schlitz = new DomesticBeer();
        Beer we_love_schlitz = new DomesticBeer("Sam Adams", 1.0);
        truck.load(new Beer[]{schlitz, more_schlitz, we_love_schlitz});
        conn.makePersistent(truck);
        tx.commit();

        tx = kit.getTransaction(conn);
        tx.begin();
        truck = (Truck<Beer>) conn.getObjectByIdentity(conn.getIdentity(truck));
        truck.unload();
        tx.commit();

        tx = kit.getTransaction(conn);
        tx.begin();
        truck = (Truck<Beer>) conn.getObjectByIdentity(conn.getIdentity(truck));
        assertEquals(0, truck.unload().size());
        tx.commit();

        conn.close();
    }

    public void testDeliverBeer() throws Exception {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);

        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        Beer schlitz = new DomesticBeer();
        Beer more_schlitz = new DomesticBeer();
        Beer sams = new DomesticBeer("Sam Adams", 1.5);
        Beer caffreys = new ImportedBeer();
        truck.load(new Beer[]{schlitz, more_schlitz, sams, caffreys});
        conn.makePersistent(truck);
        Pub foobar = new Pub("Foobar");
        conn.makePersistent(foobar);
        tx.commit();

        tx = kit.getTransaction(conn);
        tx.begin();
        truck = (Truck<Beer>) conn.getObjectByIdentity(conn.getIdentity(truck));
        foobar = (Pub) conn.getObjectByIdentity(conn.getIdentity(foobar));
        foobar.addStock(truck.unload());
        tx.commit();
        conn.close();

        Set onTap = foobar.getBeersOnTap();
        assertEquals(3, onTap.size()); // Caffrey's 70, Schlitz and Sam Adams
    }

    public void testQueryExtent() throws Exception {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);

        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        Beer schlitz = new DomesticBeer();
        Beer cafs = new ImportedBeer();
        Beer sams = new DomesticBeer("Sam Adams", 1.0);
        truck.load(new Beer[]{schlitz, cafs, sams});
        conn.makePersistent(truck);
        tx.commit();

        tx = kit.getTransaction(conn);
        tx.begin();
        Query bq = QueryFactory.newQuery(Beer.class, new Criteria());
        Collection all = conn.getCollectionByQuery(bq);
        assertEquals(3, all.size());
        tx.commit();
        conn.close();
    }

    public void testQueryCollections() throws Exception {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        Beer schlitz = new DomesticBeer();
        Beer cafs = new ImportedBeer();
        Beer sams = new DomesticBeer("Sam Adams", 1.0);
        truck.load(new Beer[]{schlitz, cafs, sams});
        conn.makePersistent(truck);
        tx.commit();

        tx = kit.getTransaction(conn);
        tx.begin();
        Query bq = QueryFactory.newQuery(Truck.class, new Criteria());
        Collection all = conn.getCollectionByQuery(bq);
        for (Iterator iterator = all.iterator(); iterator.hasNext();) {
            Object o = iterator.next();
            System.err.println(o);
        }
        tx.commit();
        conn.close();
    }

    public void testReportQuery() throws Exception {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        Beer schlitz = new DomesticBeer();
        Beer cafs = new ImportedBeer();
        Beer sams = new DomesticBeer("Sam Adams", 1.0);
        truck.load(new Beer[]{schlitz, cafs, sams});
        conn.makePersistent(truck);
        tx.commit();

        ReportQueryByCriteria query = QueryFactory.newReportQuery(Beer.class, new Criteria());
        query.setAttributes(new String[]{"brand", "price"});

        conn.close();
        PersistenceBroker broker = PersistenceBrokerFactory.defaultPersistenceBroker();
        query = QueryFactory.newReportQuery(Beer.class, new String[]{"brand", "price"}, new Criteria(), false);

        Iterator results = broker.getReportQueryIteratorByQuery(query);
        while (results.hasNext()) {
            Object[] o = (Object[]) results.next();
            System.err.println(o[0] + "  " + o[1]);
        }
        broker.close();
    }
}
