
import org.apache.ojb.broker.PBKey;
import org.apache.ojb.broker.PersistenceBroker;
import org.apache.ojb.broker.PersistenceBrokerFactory;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.Query;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.otm.Kit;
import org.apache.ojb.otm.OTMConnection;
import org.apache.ojb.otm.core.Transaction;
import org.apache.ojb.otm.kit.SimpleKit;
import org.apache.ojb.otm.lock.LockType;
import org.apache.ojb.otm.lock.LockingException;
import org.odmg.OQLQuery;

import java.util.Collection;
import java.util.Iterator;

public class Deliverators {
    private static final PBKey key = PersistenceBrokerFactory.getDefaultKey();
    private static Kit kit = SimpleKit.getInstance();

    public static void main(String[] args) throws Exception {
        Pub pub = makeMyPub("T Bar");
        System.out.println(pub.getName() +
                           " is new and owes $" +
                           pub.getDebt());

        System.out.println("But has " +
                           pub.getBeersOnTap().size() +
                           " beers on tap");

        System.out.println("Let's get them some beer...");

        Truck truck = buyTheTruck();

        fillHerUp(truck);

        System.out.println("Found " + findTheCaffreys().size() +
                           " Good Beers en route to pub");

        System.out.println("Found " + findTheSchlitz().size() +
                           " Other Beers en route to pub");

        deliverEverythingTo(pub);

        System.out.println("Now there's something on tap:");
        Iterator<Beer> on_tap = pub.getBeersOnTap().iterator();
        while (on_tap.hasNext()) {
            System.out.println("  " + on_tap.next().getBrand());
        }

        System.out.println("But " + pub.getName() +
                           " needs to sell enough beer" +
                           " to pay back $" + pub.getDebt() +
                           " owed to the deliverators");

        junkTheTrucks();
        junkThePubs();
    }

    /**
     * Create via Persistence Broker
     */
    public static Pub makeMyPub(String name) {
        Pub pub = new Pub(name);
        PersistenceBroker broker =
                PersistenceBrokerFactory.createPersistenceBroker(key);
        broker.store(pub);
        broker.close();
        return pub;
    }

    /**
     * Create via OTM
     */
    public static Truck buyTheTruck() throws LockingException {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Truck<Beer> truck = new Truck<Beer>();
        conn.makePersistent(truck);
        tx.commit();
        conn.close();
        return truck;
    }


    /**
     * Create a bunch of dependent objects (BEER!) / Update Truck
     */
    public static void fillHerUp(Truck truck) throws LockingException {
        Beer schlitz = new DomesticBeer();
        Beer more_schlitz = new DomesticBeer();
        Beer good_beer =
                new ImportedBeer("Caffreys 70", "Ireland", 2.50, 0.25);

        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Truck tx_truck =
                (Truck) conn.getObjectByIdentity(conn.getIdentity(truck));
        tx_truck.load(new Beer[]{schlitz, more_schlitz, good_beer});
        tx.commit();
        conn.close();
    }

    /**
     * Find the Caffrey's via an OTM Query By Criteria
     */
    public static Collection findTheCaffreys() {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Criteria criteria = new Criteria();
        criteria.addLike("brand", "Caffreys%");
        Query query = QueryFactory.newQuery(Beer.class, criteria);
        Collection beers = conn.getCollectionByQuery(query);
        tx.commit();
        conn.close();
        return beers;
    }

    /**
     * Find the Caffrey's via an OTM OQl Query
     */
    public static Iterator findTheCaffreysAgain() throws Exception {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        OQLQuery query = conn.newOQLQuery();
        query.create("select beer from Beer where brand like $1");
        query.bind(new Object[]{"Caffreys%"});
        Iterator beers = conn.getIteratorByOQLQuery(query);
        tx.commit();
        conn.close();
        return beers;
    }

    /**
     * Find the Schlitz via a Persistence Broker query
     */
    public static Collection findTheSchlitz() {
        PersistenceBroker broker =
                PersistenceBrokerFactory.createPersistenceBroker(key);
        Criteria criteria = new Criteria();
        criteria.addLike("brand", "Schlitz");
        Query query = QueryFactory.newQuery(Beer.class, criteria);
        Collection beers = broker.getCollectionByQuery(query);
        broker.close();
        return beers;
    }

    /**
     * Update lots of transactional objects
     */
    public static void deliverEverythingTo(Pub pub) throws LockingException {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);

        tx.begin();
        Query query = QueryFactory.newQuery(Truck.class, new Criteria());
        Collection all_trucks = conn.getCollectionByQuery(query,
                                                          LockType.WRITE_LOCK);
        Pub tx_pub = (Pub) conn.getObjectByIdentity(conn.getIdentity(pub),
                                                    LockType.WRITE_LOCK);
        for (Iterator iterator = all_trucks.iterator(); iterator.hasNext();) {
            Truck<Beer> truck = (Truck<Beer>) iterator.next();
            tx_pub.addStock(truck.unload());
        }
        tx.commit();
        conn.close();
    }

    /**
     * Really inefficient mass deletion
     */
    public static void junkTheTrucks() throws LockingException {
        OTMConnection conn = kit.acquireConnection(key);
        Transaction tx = kit.getTransaction(conn);
        tx.begin();
        Query query = QueryFactory.newQuery(Truck.class, new Criteria());
        Collection trucks = conn.getCollectionByQuery(query);
        for (Iterator iterator = trucks.iterator(); iterator.hasNext();) {
            Truck truck = (Truck) iterator.next();
            conn.deletePersistent(truck);
        }
        tx.commit();
        conn.close();
    }

    /**
     * Very efficient mass deletion
     */
    public static void junkThePubs() {
        PersistenceBroker broker =
                PersistenceBrokerFactory.createPersistenceBroker(key);
        Query query = QueryFactory.newQuery(Pub.class, new Criteria());
        broker.beginTransaction();
        broker.deleteByQuery(query);
        broker.commitTransaction();
        broker.close();
    }
}
